/***************************************************************************
 *
 * knetworkmanager-tray.cpp - A NetworkManager frontend for KDE
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Timo Hoenig        <thoenig@suse.de>, <thoenig@nouse.net>
 *         Will Stephenson    <wstephenson@suse.de>, <wstephenson@kde.org>
 *         Valentine Sinitsyn <e_val@inbox.ru>
 *         Helmut Schaa       <hschaa@suse.de>, <helmut.schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

class WirelessDialog;

#include "knetworkmanager-dialogfab.h"

#include <iwlib.h>

#include <qevent.h>
#include <qvbox.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qpixmap.h>
#include <qpixmapcache.h>
#include <qpainter.h>
#include <qstyle.h>
#include <dcopclient.h>
#include <kdebug.h>
#include <kdialogbase.h>
#include <knotifyclient.h>
#include <knotifydialog.h>
#include <klocale.h>
#include <kstdguiitem.h>
#include <khelpmenu.h>
#include <kprocess.h>
#include <kiconloader.h>
#include <kconfig.h>
#include <kmessagebox.h>

#include "knetworkmanager-ui-networklistview.h"
#include "knetworkmanager-storage.h"
#include "knetworkmanager-vpnconnectionsdialog.h"
#include "knetworkmanager-connectioninfodialog.h"
#include "knetworkmanager-settings.h"

#define KDED_NETWORK_NAME "NMNetwork"

Subhead::Subhead (QWidget* parent, const char* name, const QString & caption, const QPixmap icon)
       : QWidget (parent, name)
{
	QBoxLayout* l        = new QHBoxLayout (this);
	QLabel* lbl_icon     = new QLabel (NULL, this);
	QLabel* lbl_caption  = new QLabel (caption, this);
	QSpacerItem* sp_item = new QSpacerItem (20, 10);

	lbl_icon->setPixmap (icon);

	l->addWidget (lbl_icon);
	l->addWidget (lbl_caption);
	l->addItem (sp_item);
	l->activate ();
}

Subhead::~Subhead()
{

}

void
WirelessNetworkItem::paint (QPainter *p, const QColorGroup &/*cg*/, bool highlighted, bool /*enabled*/, int x, int y, int w, int h)
{
	int spacer = 0;
	pbarStrength->setTotalSteps (100);
	pbarStrength->setSizePolicy (QSizePolicy ((QSizePolicy::SizeType) 0, (QSizePolicy::SizeType) 0, 0, 0, pbarStrength->sizePolicy ().hasHeightForWidth ()));
	pbarStrength->setFixedHeight (_height - 2);
	pbarStrength->setProgress (_strength);
	pbarStrength->setPercentageVisible (false);

	if (highlighted) {
		p->setPen(parent->palette ().active ().highlightedText ());
		pbarStrength->setBackgroundMode( Qt::PaletteHighlight, Qt::PaletteHighlight );
	} else {
		p->setPen(parent->palette ().active ().text ());
	}
	p->drawText   (x, y, w, h, AlignLeft | AlignVCenter | DontClip | ShowPrefix, _essid);

	if (_encrypted) {
		QPixmap pmLock = SmallIcon ("lock", QIconSet::Automatic);
		p->drawPixmap (parent->sizeHint ().width () - pbarStrength->width () - _space - pmLock.size ().width () - 6, y + _border, pmLock);
		spacer = pmLock.size ().width () + 4;
	}

	if (_adhoc) {
		QPixmap pmAdHoc = SmallIcon ("system", QIconSet::Automatic);
		p->drawPixmap (parent->sizeHint ().width () - pbarStrength->width () - _space - pmAdHoc.size ().width () - spacer - 6, y + _border, pmAdHoc);
	} 
	
	p->drawPixmap (parent->sizeHint ().width () - pbarStrength->width () - _space, y +_border, QPixmap::grabWidget (pbarStrength));
}

QSize
WirelessNetworkItem::sizeHint ()
{
	return QSize (_width, _height);
}

WirelessNetworkItem::WirelessNetworkItem (QWidget* p, const QString &essid, bool /*active*/, int strength, bool encrypted, bool adhoc)
		       : QCustomMenuItem ()
{
	parent       = p;
	_essid       = essid;
	_strength    = strength;
	_encrypted   = encrypted;
	_adhoc       = adhoc;
	pbarStrength = new QProgressBar (this->parent, "pbarStrength");
	pbarStrength->hide();
	// TODO make all this neat or at least static
	/* px between pmLock and pbarStrength */
	_space     = 5;

	/* the following code should not be used as teaching material in any kind */
	// set the widget height to that of a small icon
	_widgetHeight = SmallIcon (NULL, QIconSet::Automatic).height ();
	// figure out the height of the text
	_textHeight = kapp->fontMetrics ().size (AlignLeft | AlignVCenter | DontClip | ShowPrefix, _essid).height ();
	// the item is height of the text plus the default frame width
	_height = _textHeight + kapp->style ().pixelMetric (QStyle::PM_DefaultFrameWidth);
	/* _height needs to be at least the height of a SmallIcon plus a border of one px (top & bottom)*/
	if (_height < _widgetHeight) {
		_height = _widgetHeight + 2;
	}

	/* border required for pmLock and pbarStrength */
	_border = 1;
	if (_border == kapp->style ().pixelMetric (QStyle::PM_DefaultFrameWidth))
		_border += 2 * kapp->style ().pixelMetric (QStyle::PM_DefaultFrameWidth);
	if (_border == 0)
		_border ++;

	_width =  kapp->fontMetrics().size(AlignLeft | AlignVCenter | DontClip | ShowPrefix, _essid).width(); /* essid */
	_width += SmallIcon (NULL, QIconSet::Automatic).width ();                                             /* pmLock */
	_width += _space;                                                                                     /* pmLock <-> pbarStrength */
	_width += 100;                                                                                        /* pbarStrength */
	_width += _space;                                                                                     /* pbarStrength -> contextMenu */
}

WirelessNetworkItem::~WirelessNetworkItem ()
{
}

void
Tray::updateTooltip ()
{
	DeviceStore* store = _ctx->getDeviceStore ();
	State*       state = _ctx->getState ();

	if (state->isNetworkManagerRunning ()) {
		if (state->isConnected ()) {
			Device* dev = store->getActiveDevice ();
			if (dev) {
				if (dev->isWired ()) {
					_tooltip = i18n ("Wired ethernet connection active: ") + dev->getInterface ();
				} else if (dev->isWireless ()) {
					Network* net = store->getActiveNetwork (dev);
					_tooltip = i18n ("Wireless ethernet connection active: ") + dev->getInterface ();
					if (net) {
						_tooltip += " (" + i18n ("active network: ") + net->getEssid () + ")";
					} else {
						_tooltip += " " + i18n ("Warning: No active wireless network found.");
					}
				}
			}
		} else if (state->isSleeping ()) {
			_tooltip = i18n ("Offline Mode is currently active");
		} else if (state->isConnecting ()) {
			/* FIXME string composed strange due to string freeze */
			_tooltip = i18n ("Connection in progress");
			Device* dev = store->getActiveDevice ();
			if (dev) {
				if (dev->isWired ()) {
					_tooltip += ": " + dev->getInterface ();
				} else if (dev->isWireless ()) {
					Network* net = store->getActiveNetwork (dev);
					if (net)
						_tooltip += ": " + net->getEssid () + " (" + dev->getInterface () + ")";
				}
			}
		} else if (state->isDialupActive ()) {
			_tooltip = i18n ("Dial-Up Network"); 
		} else if (state->isDisconnected ()) {
			_tooltip = i18n ("Disconnected");
		}
	} else {
		_tooltip = i18n ("NetworkManager is not running.");
	}

	return;
}

void
Tray::enterEvent (QEvent* /*e*/)
{
	updateTooltip ();
	QToolTip::add (this, _tooltip);
}

void
Tray::drawContents (QPainter *p)
{
	QRect r = contentsRect();

	if (m_animated) {
		if (!testWFlags(Qt::WNoAutoErase)) { // We are rendering a first frame - grab the background
			setWFlags(Qt::WNoAutoErase);
			erase(r); // Be sure we grab the background, not anything else
			m_background = QPixmap::grabWindow( this->winId(), r.x(), r.y(), r.width(), r.height() );
		}

		QPixmap backBuffer(m_background);
		QPainter back(&backBuffer);

		back.translate(-r.x(), -r.y()); // KSystemTray::drawContents() draws inside contentsRect() only, but we are already inside.
		KSystemTray::drawContents(&back); 
		bitBlt(this, r.topLeft(), &backBuffer);
	} else {
		if (testWFlags(Qt::WNoAutoErase)) { // First non-animated frame - revert to normal rendering
			clearWFlags(Qt::WNoAutoErase);
			erase(r);
		}

		KSystemTray::drawContents(p);
		// draw overlay image
		if (!m_overlay.isNull())
			p->drawPixmap(2,r.height()/2-2,m_overlay);
	}
}

void
Tray::connectHiddenNetwork ()
{
	NewWirelessNetworkDialog* dlg;
	dlg = new NewWirelessNetworkDialog (this, "connecthiddennetworkdialog", false, 0, _ctx);
	dlg->show ();
}

void
Tray::configureVPN ()
{
	VPNConnectionsDialog* mvc;
	mvc = dynamic_cast<VPNConnectionsDialog*>(this->child("VPNConnectionsDialog", "VPNConnectionsDialog"));
	if (!mvc)
		mvc = new VPNConnectionsDialog(_ctx, this, "VPNConnectionsDialog", FALSE, Qt::WDestructiveClose);
	
	// we want to notify NM when connections are changed
	connect(mvc, SIGNAL(done()), _ctx->getVPN(), SLOT(updateVPNConnections()));
	
	// show the new Dialog
	mvc->show();
	mvc->raise();
	mvc->setActiveWindow();
}

void
Tray::disconnectVPN ()
{
	emit disconnectVPNConnection ();
}

void
Tray::slotQuit()
{
	KConfig * config = KGlobal::config();
	config->setGroup("Notification Messages");
	if (config->readEntry("AutostartDontAskAgain").isEmpty()) {
		int answer = KMessageBox::questionYesNo (
		  0,
		  i18n ("Start KNetworkManager automatically when you log in?"),
		  i18n ("Question"),
		  i18n ("Start Automatically"),
		  i18n ("Do Not Start"),
		  "AutostartDontAskAgain");

		config->setGroup ("General");
		config->writeEntry ("Autostart", answer == KMessageBox::Yes);
		config->sync ();
	}
}

void
Tray::configureDialUp ()
{
#ifdef KNETWORKMANAGER_DIALUP_CONFIG
	KProcess* dialupConfig = new KProcess ();
	QStringList command = QStringList::split (" ", KNETWORKMANAGER_DIALUP_CONFIG, false);
	for (QStringList::Iterator it = command.begin (); it != command.end (); ++it)
		*dialupConfig << (*it);
	dialupConfig->start ();
#endif
}

void
Tray::configureNotifications ()
{
	KNotifyDialog::configure (this);
}

void
Tray::switchState ()
{
	State* state    = _ctx->getState ();
	bool   newState = !state->isSleeping ();

	state->setOfflineMode (newState);
	KNetworkManagerStorage::getInstance ()->persistOfflineMode (newState);
}

void
Tray::switchWirelessOn ()
{
	_ctx->getState ()->setWirelessState (true);
	KNetworkManagerStorage::getInstance ()->persistWireless (true);
}

void
Tray::switchWirelessOff ()
{
	_ctx->getState ()->setWirelessState (false);
	KNetworkManagerStorage::getInstance ()->persistWireless (false);
}

void
Tray::showNetworksDialog()
{
	KDialogBase * dialog = new KDialogBase( this, "NetworksDialog", true, i18n( "Manage Wireless Networks" ), KDialogBase::Ok|KDialogBase::Cancel );
	dialog->makeVBoxMainWidget();

	KNetworkManagerNetworkListView * lv = new KNetworkManagerNetworkListView( _ctx->getNetworkManagerInfo(), dialog->mainWidget(), "networklv" );
	QToolTip::add( lv, i18n( "Drag and drop to move networks between groups" ) );
	//QHBox * buttonBox = new QHBox( dialog->mainWidget() );
	QWidget * wid = new QWidget( dialog->mainWidget() );
	QHBoxLayout * vbox = new QHBoxLayout( wid );
	vbox->setSpacing( 4 );
	QPushButton * addBtn = new QPushButton( i18n("Add Network..." ), wid );
	QPushButton * addApBtn = new QPushButton( i18n("Add Access Point..." ), wid );
	QPushButton * remBtn = new QPushButton( i18n("Remove Item" ), wid );
	QToolTip::add( addBtn, i18n( "Add a new network to the selected group" ) );
	QToolTip::add( addApBtn, i18n( "Add a new access point to the selected network" ) );
	QToolTip::add( remBtn, i18n( "Remove the selected network or access point" ) );
	vbox->addWidget( addBtn );
	vbox->addWidget( addApBtn );
	vbox->addStretch();
	vbox->addWidget( remBtn );
	
	connect( addBtn, SIGNAL( clicked() ), lv, SLOT( slotAddNetworkClicked() ) );
	connect( addApBtn, SIGNAL( clicked() ), lv, SLOT( slotAddAccessPointClicked() ) );
	connect( remBtn, SIGNAL( clicked() ), lv, SLOT( slotRemoveItemClicked() ) );
	
	  void slotAddNetworkClicked();
	  void slotAddAccessPointClicked();
	  void slotRemoveItemClicked();

	addBtn->setEnabled( false );
	addApBtn->setEnabled( false );
	addBtn->setHidden( true );
	addApBtn->setHidden( true );
	if ( dialog->exec() == QDialog::Accepted )
	{
		KNetworkManagerStorage * storage = KNetworkManagerStorage::getInstance();
		// changed networks
		QValueList< Network * > changes = lv->changedNetworks();
		QValueList< Network * >::Iterator it = changes.begin();
		for ( ; it != changes.end(); ++it )
			storage->storeNetwork( *it, false );
		// deleted networks
		QValueList< Network * > deleted = lv->deletedNetworks();
		QValueList< Network * >::Iterator delIt = deleted.begin();
		for ( ; delIt != deleted.end(); ++delIt )
			storage->removeNetwork( *delIt );
	}
}

void
Tray::itemHighlighted (int id)
{
	Device* dev = _ctx->getDeviceStore()->getDevice(_deviceMap[id]);
	Network* net = NULL;
	if (_networkMap[id])
		net = dev->getNetwork(_networkMap[id]);
	QString tooltip;

	if (dev && !net) {
		int speed = dev->getSpeed ();

		tooltip = dev->getVendor () + " "	+ dev->getProduct () 								+ "\n";
		tooltip += i18n ("Device: ")            + dev->getInterface ()								+ "\n";
		tooltip += i18n ("IP: ")                + dev->getIPv4Address ()							+ "\n";
		tooltip += i18n ("Hardware address: ")  + dev->getHardwareAddress ()							+ "\n";
		tooltip += i18n ("Subnet mask: ")       + dev->getSubnetmask ()								+ "\n";
		tooltip += i18n ("Broadcast: ")         + dev->getBroadcast ()								+ "\n";
		tooltip += i18n ("Active: ")            + (dev->isActive () ? i18n ("yes") : i18n ("no"))				+ "\n";
		tooltip += i18n ("Carrier detect: ")    + (dev->hasCarrierDetect () ? i18n ("supported") : i18n ("not supported"))	+ "\n";
		if (speed)
			tooltip += i18n ("Bandwidth: " )+ QString ("%1").arg (speed) + " Mb/s ";
	} else if (net) {
		DeviceStore* store       = _ctx->getDeviceStore ();
		bool         isEncrypted = net->isEncrypted ();

		dev = store->getDevice (net);
                if (dev) {
			int     signalStrength = net->getStrength () ? net->getStrength () : dev->getStrength ();
			int     speed          = net->getRate () / 1024;
			if (speed == 0) speed = dev->getSpeed();
			double  frequency      = net->getFrequency ();

			tooltip = dev->getVendor () + " "      + dev->getProduct ()								+ "\n";
			tooltip += i18n ("Device: ")           + dev->getInterface ()								+ "\n";
			tooltip += i18n ("IP: ")               + dev->getIPv4Address ()								+ "\n";
			tooltip += i18n ("Hardware address: ") + dev->getHardwareAddress ()							+ "\n";
			tooltip += i18n ("Subnetmask: ")       + dev->getSubnetmask ()								+ "\n";
			tooltip += i18n ("Broadcast: ")        + dev->getBroadcast ()								+ "\n";
			tooltip += i18n ("Network: ")          + net->getEssid ()								+ "\n";
			tooltip += i18n ("Hidden: ")	       + QString ("%1").arg (net->isHidden () ? i18n ("yes") : i18n ("no"))		+ "\n";
			tooltip += i18n ("Access point: ")     + net->getHardwareAddresses ().join(", ")					+ "\n";
			tooltip += i18n ("Bandwidth: ");
			if (speed)
				tooltip += QString ("%1").arg (speed) + QString (" Mb/s")							+ "\n";
			else
				tooltip += i18n ("Unknown")											+ "\n";
			tooltip += i18n ("Signal strength: ")  + QString ("%1").arg (signalStrength)						+ "\n";
			tooltip += i18n ("Frequency: ");
			if (frequency)

				tooltip += QString ("%1").arg (frequency / 1000000000, 0, 'f', 3) + " GHz"					+ "\n";
			else
				tooltip += i18n ("Unknown")											+ "\n";
			tooltip += i18n ("Encrypted: ")        + QString ("%1").arg (isEncrypted ? i18n ("yes") : i18n ("no"))			+ "\n";
			if (isEncrypted) {
				QStringList encryptionProtocol = net->getEncryptionProtocol ();
				tooltip += i18n ("Encryption protocol: ");
				for (QStringList::Iterator it = encryptionProtocol.begin (); it != encryptionProtocol.end (); ++it)
					 tooltip += (*it) + " ";
				tooltip += "\n";
			}
			tooltip += i18n ("Device active: ")    + QString ("%1").arg (dev->isActive () ? i18n ("yes") : i18n ("no"))		+ "\n";
			tooltip += i18n ("Network active: ")   + QString ("%1").arg (net->isActive () ? i18n ("yes") : i18n ("no"));
		}
	}

	QToolTip::remove (contextMenu ());
	QToolTip::add (contextMenu (), tooltip);
}

void
Tray::itemActivated (int id)
{
	Device* dev = _ctx->getDeviceStore()->getDevice(_deviceMap[id]);
	Network* net = NULL;
	if (_networkMap[id])
		net = dev->getNetwork(_networkMap[id]);
	ActivationStageNotifyNetwork* notify;

	if (!dev && !net) {
		return;
	}

	if (dev && !net) {
		QString iface = dev->getInterface ();
		emit activateDevice (dev);
	} else if (net) {
		if (!dev) {
			if (NULL == (dev = _ctx->getDeviceStore ()->getDevice (net)))
				return;
		}

		emit activateNetwork (net, dev);
	}

	/* destroy an activation stage notify dialogs and create a new one */
	emit destroyActivationStage ();
	notify = new ActivationStageNotifyNetwork ( net ? net->getEssid() : QString::null, this, "ActivationStageNotify", false, 0, _ctx, dev );
	notify->setAnchor(_anchor);
	notify->show();
}

void
Tray::vpnItemActivated (int id)
{
	VPNConnection* vpnConnection = _vpnMap [id];
	ActivationStageNotifyVPN* notify;

	if (!vpnConnection)
		return;

	/* destroy an activation stage notify dialogs and create a new one */
	emit destroyActivationStage ();
	notify = new ActivationStageNotifyVPN (this, "ActivationStageNotify", false, 0, _ctx, vpnConnection);
	notify->setAnchor(_anchor);
	notify->show();
	emit activateVPNConnection (vpnConnection);
}

void
Tray::addVPNConnection (VPNConnection* vpnConnection, bool state)
{
	int id;
	QString item = QString::null;
	QString prefix = QString::null;
	QString postfix = QString::null;
	QString icon("encrypted");

	VPNService* service = vpnConnection->getVPNService();
	if (service)
		icon = service->getIcon();

	NMVPNActStage stage = vpnConnection->getActivationStage ();

	if (stage == NM_VPN_ACT_STAGE_UNKNOWN) {
		postfix = i18n ("unknown state");
	} else if (stage == NM_VPN_ACT_STAGE_DISCONNECTED) {
		prefix = i18n ("Connect to");
	} else if (stage == NM_VPN_ACT_STAGE_PREPARE) {
		postfix = i18n ("preparing");
	} else if (stage == NM_VPN_ACT_STAGE_CONNECT) {
		postfix = i18n ("connecting");
	} else if (stage == NM_VPN_ACT_STAGE_IP_CONFIG_GET) {
		postfix = i18n ("IP configuration");
	} else if (stage == NM_VPN_ACT_STAGE_ACTIVATED) {
		postfix = i18n ("active");
	} else if (stage == NM_VPN_ACT_STAGE_FAILED) {
		postfix = i18n ("failed");
	} else if (stage == NM_VPN_ACT_STAGE_CANCELED) {
		postfix = i18n ("canceled");
	}

	if (prefix != QString::null)
		item += prefix + " ";
	item += vpnConnection->getName ();
	if (postfix != QString::null)
		item += " (" + postfix + ")";

	id = _vpnMenu->insertItem (SmallIcon (icon, QIconSet::Automatic), item, -1, -1);
	_vpnMenu->setItemEnabled (id, state);

	if (stage == NM_VPN_ACT_STAGE_ACTIVATED)
		_vpnMenu->setItemChecked (id, true);
	else
		_vpnMenu->setItemChecked (id, false);
	
	_vpnMap [id] = vpnConnection;
}

void
Tray::dialUpItemActivated (int id)
{
	DialUp* dialup = _dialUpMap [id];

	if (dialup) {
		if (dialup->isActive ()) {
			emit deactivateDialUp (dialup);
			_ctx->getState ()->setDialupState (false);
			slotStateChanged ();
		} else {
			emit activateDialUp (dialup);
			_ctx->getState ()->setDialupState (true);
			slotStateChanged ();
		}
	}
}

void
Tray::addDialUp (DialUp* dialup)
{
	int id;
	QString item = QString::null;

	item = dialup->isActive () ? i18n ("Hang up ") : i18n ("Connect to ");

	item += dialup->getName ();
	id = _dialUpMenu->insertItem (SmallIcon ("modem",  QIconSet::Automatic), item, -1, -1);
	_dialUpMap [id] = dialup;
}

void
Tray::addWirelessNetwork (Device* dev, Network* net)
{
	QPixmap icon;
	QString descr;
	WirelessNetworkItem* wirelessNetworkItem;
	bool online = _ctx->getState ()-> isConnected ();
	int id;

	if (!net->getEssid ()) {
		printf ("Refusing to add network without known ESSID\n");
		return;
	}

	wirelessNetworkItem = new WirelessNetworkItem (contextMenu (),
						       net->getEssid (),
						       net->isActive () && dev->isActive (),
						       net->getStrength () ? net->getStrength () : dev->getStrength (),
						       net->isEncrypted (),
						       (net->getMode () == IW_MODE_INFRA) ? false : true);
	id = contextMenu ()->insertItem (wirelessNetworkItem, -1, -1);

	contextMenu ()->setItemChecked (id, online && net->isActive () && dev->isActive ());

	_networkMap[id] = net->getObjectPath();
	_deviceMap[id] = dev->getObjectPath();
}

void
Tray::addWiredDevice (Device* dev, const QString & descr)
{
	QPixmap icon;
	bool online = _ctx->getState ()-> isConnected ();
	int id;

	id = contextMenu ()->insertItem (descr, -1, -1);

	contextMenu ()->setItemChecked (id, online && dev->isActive ());
	contextMenu ()->setItemEnabled (id, dev->getLinkActive ());

	_deviceMap[id] = dev->getObjectPath();
}

void
Tray::addWirelessNetworks (Device* dev)
{
	NetworkList::iterator i;
	NetworkList networkList = dev->getNetworkList ();

	if (networkList.empty ()) {
		QLabel* notFound = new QLabel (i18n ("<i>No wireless network found.</i>"), 0);
		notFound->setAlignment (Qt::AlignHCenter | Qt::AlignVCenter);
		contextMenu ()->insertItem (notFound); //reparents notFound to the context menu
		return;
	}
	for (i = networkList.begin (); i != networkList.end (); ++i) {
		addWirelessNetwork (dev, *i);
	}
}

void
Tray::addSubheading (KPopupMenu* menu, QWidget* parent, const QString & caption, const QPixmap icon)
{
	Subhead* subhead = new Subhead (parent, "subhead", caption, icon);
	menu->insertItem (subhead, -1, -1);
}

void
Tray::contextMenuAboutToShow (KPopupMenu* menu)
{
	DeviceStore* store      = _ctx->getDeviceStore ();
	State*       state      = _ctx->getState ();
	VPN*         vpn        = _ctx->getVPN ();
	DeviceList  deviceList = store->getDeviceList ();
	DialUpList  dialUpList = store->getDialUpList ();
	VPNList*     vpnList    = vpn->getVPNList ();

	s_status*    stat       = store->getStatus ();

	bool         devicesFound  = false;
	bool         networksFound = false;
	bool         separated = false;

/*	printf ("STATUS: Wired: %i, wireless: %i (networks: %i)\n", stat->wiredDevices,
								    stat->wirelessDevices,
								    stat->wirelessNetworks); */
	menu->clear ();
	
	_vpnMenu->clear ();
	_dialUpMenu->clear ();
	_optionsMenu->clear ();
	
	_vpnMap.clear ();
	_dialUpMap.clear ();
	_deviceMap.clear ();

	menu->insertTitle (SmallIcon ("knetworkmanager",  QIconSet::Automatic), "KNetworkManager", -1, -1);

	if (!state->isNetworkManagerRunning ()) {
		QLabel* nmNotRunning= new QLabel (i18n ("NetworkManager is not running."), 0);

		nmNotRunning->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
		menu->insertItem (nmNotRunning);
		menu->insertSeparator ();
	} else if (state->isSleeping ()) {
		QLabel* offlineMode= new QLabel (i18n ("<b>Offline Mode is currently active.<b>"), 0);

		offlineMode->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
		int id = contextMenu ()->insertItem (offlineMode);
		menu->setItemEnabled (id, false);
	} else if (state->isWaitingForKey()) {
		QLabel* awaitingKey = new QLabel (i18n ("<b>Waiting for key from KWallet.<b>"), 0);

		awaitingKey->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
		int id = contextMenu ()->insertItem (awaitingKey);
		menu->setItemEnabled (id, false);
		
		id = contextMenu()->insertItem(SmallIcon ("cancel",  QIconSet::Automatic), i18n ("Cancel key request"), KNetworkManagerStorage::getInstance(), SLOT(slotCancelAllCredentialRequests()));
		menu->insertSeparator();

	} else {
		/* add wired devices and a separator */
		if (stat->wiredDevices) {
			DeviceList::iterator i;
			QString descr;

			devicesFound = true;
			addSubheading (menu, this, i18n("Wired Devices"), SmallIcon ("wired"));

			if (stat->wiredDevices == 1) {
				for (i = deviceList.begin (); i != deviceList.end (); ++i) {
					if ((*i)->isWired ()) {
						descr = i18n ("Wired Network");
						addWiredDevice (*i, descr);
					}
				}
			} else {
				for (i = deviceList.begin (); i != deviceList.end (); ++i) {
					if ((*i)->isWired ()){
						descr = (*i)->getVendor () + " " + (*i)->getProduct () + " (" + (*i)->getInterface () + ")";
						addWiredDevice (*i, descr);
					}
				}
			}
			menu->insertSeparator ();
		}

		/* add wireless devices, their networks (if applicable), connect to hidden network item and a separator */
		if (stat->wirelessDevices && state->isWirelessEnabled ()) {
			devicesFound = true;
			addSubheading (menu, this,  i18n ("Wireless Networks"), SmallIcon ("wireless"));
			for (DeviceList::iterator i = deviceList.begin (); i != deviceList.end (); ++i) {
				if ((*i)->isWireless ()) {
					/* only show vendor and product if system has more than one wireless device */
					if (stat->wirelessDevices > 1) {
						QString caption;
						caption += (*i)->getVendor () + " " + (*i)->getProduct () + " (" + (*i)->getInterface () + ")";
						addSubheading (menu, this, caption, NULL);
					}
					addWirelessNetworks (*i);
					networksFound = true;
				}
			}
			KAction* hiddenAction = actionCollection ()->action ("connect_hidden");
			menu->insertSeparator ();
			hiddenAction->plug (menu);
			menu->insertSeparator ();
			separated = true;
		}

		/* if we don't have any network devices make this clear */
		if (!devicesFound) {
			QLabel* noDeviceFound = new QLabel (i18n ("No network device found."), 0);
			noDeviceFound->setAlignment (Qt::AlignHCenter | Qt::AlignVCenter);
			contextMenu ()->insertItem (noDeviceFound);
		}

		if (stat->dialUpConnections) {
			/* show existing dialup connections */
			for (DialUpList::iterator i = dialUpList.begin (); i != dialUpList.end (); ++i) {
				addDialUp (*i);
			}
		}

#ifndef KNETWORKMANAGER_DIALUP_CONFIG
		if (stat->dialUpConnections)
			menu->insertItem (SmallIcon ("modem",  QIconSet::Automatic), i18n ("Dial-Up Connections"), _dialUpMenu, -1);
#else
		menu->insertItem (SmallIcon ("modem",  QIconSet::Automatic), i18n ("Dial-Up Connections"), _dialUpMenu, -1);
		separated = false;
#endif

		/* show entry for VPN if services are available */
		if (vpn->isAvailable ()) {
			bool isActive = vpn->isActive ();
			bool vpnConnectionFound = false;
			/* show existing VPN connections */
			for (VPNList::iterator i = vpnList->begin (); i != vpnList->end (); ++i) {
				/* enable items of VPN connections if (a) no other VPN connection is active
				 * and (b) we're currently connected to the internet
				 */
				if ((*i)->isConfirmedByNM())
				{
					addVPNConnection (*i, !isActive && state->isConnected ());
					vpnConnectionFound = true;
				}
			}

			if (vpnConnectionFound)
				_vpnMenu->insertSeparator ();
			
			KAction* disconnectVPNAction = actionCollection ()->action ("disconnect_vpn");
			disconnectVPNAction->setEnabled (isActive);
			disconnectVPNAction->plug (_vpnMenu);

			menu->insertItem (SmallIcon ("encrypted",  QIconSet::Automatic), i18n ("VPN Connections"), _vpnMenu, -1);
			separated = false;
		}
	}

	/* build-up the ``Options'' sub menu if NM is running */
	if (state->isNetworkManagerRunning () && !state->isWaitingForKey()) {
		if (!separated)
		    menu->insertSeparator ();
		KAction* switchWirelessOnAction  = actionCollection ()->action ("switch_wireless_on");
		KAction* switchWirelessOffAction = actionCollection ()->action ("switch_wireless_off");

		switchWirelessOnAction->setEnabled  (!state->isWirelessEnabled () && !state->isSleeping ());
		switchWirelessOffAction->setEnabled (state->isWirelessEnabled () && !state->isSleeping ());
		switchWirelessOnAction->plug  (_optionsMenu);
		switchWirelessOffAction->plug (_optionsMenu);
		_optionsMenu->insertSeparator ();

		KAction* switchOnAction = actionCollection ()->action ("switch_on");
		KAction* switchOffAction = actionCollection ()->action ("switch_off");
		switchOnAction->setEnabled  (state->isSleeping ());
		switchOffAction->setEnabled (!state->isSleeping ());
		switchOnAction->plug (_optionsMenu);
		switchOffAction->plug (_optionsMenu);

		_optionsMenu->insertSeparator ();
		actionCollection()->action("show_connection_info")->plug(_optionsMenu);
		actionCollection()->action("show_settings_dialog")->plug(_optionsMenu);

		menu->insertItem (SmallIcon ("configure",  QIconSet::Automatic), i18n ("Options"), _optionsMenu, -1);
	}

	/* Help menu */
	KHelpMenu* helpMenu = new KHelpMenu (this, KGlobal::instance ()->aboutData (), false);
	helpMenu->menu ()->removeItemAt (KHelpMenu::menuHelpContents);
	/* once the help menu is gone, remove the separator which is at position KHelpMenu::menuHelpContents now */
	helpMenu->menu ()->removeItemAt (KHelpMenu::menuHelpContents);
	menu->insertItem (SmallIcon ("help"), i18n ("&Help"), helpMenu->menu ());

	menu->insertSeparator ();
	KAction* quitAction = actionCollection ()->action (KStdAction::name (KStdAction::Quit));
	if (quitAction) {
		quitAction->plug (menu);
	}
}

void Tray::mouseMoveEvent (QMouseEvent * e)
{
	userInteraction ();
	_anchor = e->globalPos();
}

void Tray::mousePressEvent( QMouseEvent *e )
{
	if (e->button() == RightButton) {
		contextMenuAboutToShow (contextMenu());
	 	contextMenu ()->popup (e->globalPos());
	}
	else if (e->button () == LeftButton) {
		ConnectionInfoDialog* dlg = NULL;
		dlg = dynamic_cast<ConnectionInfoDialog*>(this->child("ConnectionInfoDialog", "ConnectionInfoDialog"));
		if (dlg)
			if (dlg->isHidden())
				dlg->show();
			else
				dlg->close();
		else
			slotShowConnectionInfo();
	}
}

void Tray::vpnConnectionStateChanged(bool connected)
{
	if (connected) {
		QRect r = contentsRect();
		m_overlay = SmallIcon("encrypted", r.width()*1/2);
	}
	else
		m_overlay = QPixmap();
	repaint();
}

void Tray::slotStateChanged()
{
	QPixmap pm = updateForState( _ctx->getState( )->getConnectionState( ) );

	if ( m_animated )
		m_iconTimer.start( 100 );
	else if ( m_iconTimer.isActive() )
		m_iconTimer.stop();
	if ( m_animated || pm.serialNumber() != pixmap()->serialNumber() )
		setPixmap( pm );
}

void
Tray::slotIconTimer()
{
	m_step++;
	if ( m_animated && _ctx->getState( )->getConnectionState() == NM_STATE_CONNECTING )
		setPixmap( pixmapForStage() );
}

void
Tray::loadIcons()
{
	// load all the connecting icons and the static (connected, disconnected etc icons)
	KGlobal::iconLoader()->addAppDir("knetworkmanager");
	QString templateName;
	for ( int i = 0; i < NUM_CONNECTING_STAGES; i++ )
	{
		for (int j = 0; j < NUM_CONNECTING_FRAMES; j ++ )
		{
			templateName.sprintf( "nm_stage%02i_connecting%02i", i+1, j+1 );
			m_pixmaps[i][j] = SmallIcon( templateName, 22 );
		}
	}
	m_disconnected = SmallIcon( "nm_no_connection", 22 );
	m_wired = SmallIcon( "nm_device_wired", 22 );
	m_dialup = SmallIcon ("modem", 22);
	m_adhocWireless = SmallIcon( "nm_adhoc", 22 );
	m_wireless00 = SmallIcon( "nm_signal_00", 22 );
	m_wireless25 = SmallIcon( "nm_signal_25", 22 );
	m_wireless50 = SmallIcon( "nm_signal_50", 22 );
	m_wireless75 = SmallIcon( "nm_signal_75", 22 );
	m_wireless100 = SmallIcon( "nm_signal_100", 22 );
}

QPixmap
Tray::updateForState( NMState state )
{
	QPixmap pixmap;
	Device * dev = _ctx->getDeviceStore()->getActiveDevice( );

	switch ( state )
	{
		case NM_STATE_CONNECTED:
			//FIXME: correct connected icon
			if ( !dev )
				pixmap = m_disconnected;
			else if ( dev->isWireless( ) )
			{
				int strength;
				Network* net = _ctx->getDeviceStore()->getActiveNetwork (dev);
				if (net)
					strength = net->getStrength () ? net->getStrength () : dev->getStrength ();
				else
					strength = dev->getStrength ();
				if ( strength > 80 )
					pixmap = m_wireless100;
				else if ( strength > 55 )
					pixmap = m_wireless75;
				else if ( strength > 30 )
					pixmap = m_wireless50;
				else if ( strength > 5 )
					pixmap = m_wireless25;
				else
					pixmap = m_wireless00;
			}
			else 
				pixmap = m_wired;

			m_animated = false;
			
			break;
		case NM_STATE_CONNECTING:
			if ( dev )
			{
				pixmap = pixmapForStage();
				m_animated = true;
			}
			else 
				pixmap = m_disconnected;
			break;
		case NM_STATE_ASLEEP:
		case NM_STATE_DISCONNECTED:
		default:
			if (_ctx->getState ()->isDialupActive ())
				pixmap = m_dialup;
			else
				pixmap = m_disconnected;

			m_animated = false;
			break;
	}

	return pixmap;
}

QPixmap Tray::pixmapForStage()
{
	QPixmap pixmap;
	int connectingStage = -1;
	Device* dev =  _ctx->getDeviceStore ()->getActiveDevice ();

	if (dev) {
		switch (dev->getActivationStage ())
		{
			case NM_ACT_STAGE_DEVICE_PREPARE:
			case NM_ACT_STAGE_DEVICE_CONFIG:
			case NM_ACT_STAGE_NEED_USER_KEY:
			{
				connectingStage = 0;
				break;
			}
			case NM_ACT_STAGE_IP_CONFIG_START:
			{
				connectingStage = 1;
				break;
			}
			case NM_ACT_STAGE_IP_CONFIG_GET:
			case NM_ACT_STAGE_IP_CONFIG_COMMIT:
			{
				connectingStage = 2;
				break;
			}
			default:
				break;
		}
		if (connectingStage >= 0 && connectingStage < NUM_CONNECTING_STAGES)
		{
			if (m_step >= NUM_CONNECTING_FRAMES)
				m_step = 0;
			pixmap = m_pixmaps [connectingStage][m_step];
		}
	} else {
		pixmap = m_disconnected;
	}

	return pixmap;
}

void Tray::slotLinkUp( Device * )
{
	KNotifyClient::event( winId(), "knm-nm-link-up", i18n("Link up") );
}

void Tray::slotLinkDown( Device * )
{
	KNotifyClient::event( winId(), "knm-nm-link-down", i18n("Link down") );
}

void Tray::slotDeviceAdded( Device * )
{
	KNotifyClient::event( winId(), "knm-nm-device-added", i18n("New network device found") );
}

void Tray::slotDeviceRemoved( Device * dev )
{
	//TODO: show vendor+product here
	KNotifyClient::event( winId(), "knm-nm-device-removed", i18n("Network device %1 removed").arg(dev->getInterface() ) );
}

void Tray::slotNMSleeping()
{
	KNotifyClient::event( winId(), "knm-nm-sleeping", i18n("Switched to offline mode") );
}

void Tray::slotNMConnecting()
{
	KNotifyClient::event( winId(), "knm-nm-connecting", i18n("NetworkManager is connecting" ) );
}

void Tray::slotNMConnected()
{
	KNotifyClient::event( winId(), "knm-nm-connected", i18n("NetworkManager is now connected") );
}

void Tray::slotNMDisconnected()
{
	KNotifyClient::event( winId(), "knm-nm-disconnected", i18n("NetworkManager is now disconnected") );
}

void Tray::slotNetworkFound( Network * net )
{
	KNotifyClient::event( winId(), "knm-nm-network-found", i18n("New wireless network %1 found").arg( net->getEssid() ) );
}

void Tray::slotNetworkDisappeared( Network * net )
{
	KNotifyClient::event( winId(), "knm-nm-network-gone", i18n("Wireless network %1 disappeared").arg( net->getEssid() ) );
}

QPoint
Tray::getAnchor() const
{
	return _anchor;
}

void
Tray::push (KNetworkManager* ctx)
{
	_ctx = ctx;
}

void
Tray::slotShowConnectionInfo()
{
	ConnectionInfoDialog* dlg = NULL;
	dlg = dynamic_cast<ConnectionInfoDialog*>(this->child("ConnectionInfoDialog", "ConnectionInfoDialog"));
	if (!dlg)
		dlg = new ConnectionInfoDialog(_ctx, this, 
                        "ConnectionInfoDialog", Qt::WDestructiveClose);

	// show the dialog
	dlg->show();
	dlg->raise();
	dlg->setActiveWindow();
}

void
Tray::slotShowSettingsDialog()
{
	SettingsDialog* dlg = NULL;
	dlg = dynamic_cast<SettingsDialog*>(this->child("SettingsDialog", "SettingsDialog"));
	if (!dlg)
		dlg = new SettingsDialog(_ctx, this, "SettingsDialog", Qt::WDestructiveClose, false, "Settings");
	dlg->show();
	dlg->raise();
	dlg->setActiveWindow();
}

Tray::Tray () : KSystemTray (), m_step( 0 ), m_animated( false )
{
/*	QMovie movie = KGlobal::iconLoader()->loadMovie("nm_detect", KIcon::Panel );
// 	movie.setSpeed( 3 );
	movie.setBackgroundColor( palette().active().background() );
	kdDebug() << "Movie background colour is: " << movie.backgroundColor() << endl;
	setMovie( movie	);*/
	_tooltip = QString::null;

	setPixmap (loadIcon ("knetworkmanager"));
	setMouseTracking (true);

	_vpnMenu     = new KPopupMenu (this->contextMenu (), "vpnconnmenu" );
	_dialUpMenu  = new KPopupMenu (this->contextMenu (), "dialupconnmenu" );
	_optionsMenu = new KPopupMenu (this->contextMenu (), "optionsmenu" );
	_helpMenu    = new KPopupMenu (this->contextMenu (), "helpmenu" );

	_vpnMenu->setCheckable (true);
	_optionsMenu->setCheckable (false);
	_helpMenu->setCheckable (false);
	contextMenu ()->setCheckable (true);

	connectHiddenAction   = new KAction (i18n ("Connect to Other Wireless Network..."),
					     SmallIcon ("wireless",  QIconSet::Automatic), 0,
					     this, SLOT (connectHiddenNetwork ()), actionCollection (), "connect_hidden");
	
	configureVPNAction = new KAction (i18n ("Configure VPN..."),
					  SmallIcon ("configure",  QIconSet::Automatic), 0,
					  this, SLOT (configureVPN ()), actionCollection (), "configure_vpn");

	disconnectVPNAction = new KAction (i18n ("Disconnect VPN"),
			 		   SmallIcon ("connect_no",  QIconSet::Automatic), 0,
					   this, SLOT (disconnectVPN ()), actionCollection (), "disconnect_vpn");

	configureDialUpAction = new KAction (i18n ("Configure Dial-Up Connections..."),
					     SmallIcon ("configure",  QIconSet::Automatic), 0,
					     this, SLOT (configureDialUp ()), actionCollection (), "configure_dialup");

	switchOnAction = new KAction (i18n ("Switch to Online Mode"),
					    SmallIconSet ("ok",  QIconSet::Automatic), 0,
					    this, SLOT (switchState ()), actionCollection (), "switch_on");

	switchOffAction = new KAction (i18n ("Switch to Offline Mode"),
					     SmallIconSet ("no",  QIconSet::Automatic), 0,
					     this, SLOT (switchState ()), actionCollection (), "switch_off");

	switchWirelessOnAction = new KAction (i18n ("Enable Wireless"),
					      SmallIconSet ("wireless",  QIconSet::Automatic), 0,
					      this, SLOT (switchWirelessOn ()), actionCollection (), "switch_wireless_on");

	switchWirelessOffAction = new KAction (i18n ("Disable Wireless"),
					       SmallIconSet ("wireless",  QIconSet::Automatic), 0,
					       this, SLOT (switchWirelessOff ()), actionCollection (), "switch_wireless_off");

	showNetworksDialogAction = new KAction (i18n ("Show Networks..."),
						   0, 0,
						   this, SLOT (showNetworksDialog ()), actionCollection (), "show_networks_dialog");
	showConnectionInfo = new KAction(i18n ("Show Connection Information..."),
						   SmallIcon ("info", QIconSet::Automatic), 0,
						   this, SLOT(slotShowConnectionInfo()), actionCollection(), "show_connection_info");

	showSettingsDialog = new KAction(i18n ("Configure..."),
						   SmallIcon ("package_settings", QIconSet::Automatic), 0,
						   this, SLOT(slotShowSettingsDialog()), actionCollection(), "show_settings_dialog");

	connect (this->contextMenu (), SIGNAL (activated (int)),
		 this,                 SLOT   (itemActivated (int)));

	connect (this->contextMenu (), SIGNAL (highlighted (int)),
		 this,                 SLOT   (itemHighlighted (int)));

	connect (this->_vpnMenu, SIGNAL (activated (int)),
		 this,           SLOT   (vpnItemActivated (int)));
	
	connect (this->_dialUpMenu, SIGNAL (activated (int)),
		 this,              SLOT   (dialUpItemActivated (int)));

	connect (this, SIGNAL(quitSelected()), this, SLOT(slotQuit()));

	loadIcons();

	connect ( &m_iconTimer, SIGNAL( timeout() ), SLOT( slotIconTimer() ) );
}

Tray::~Tray ()
{

}

#include "knetworkmanager-tray.moc"

