-module(ldap_domains).
-author('rgiannetto@babel.it').

-include("ejabberd.hrl").
-include("ejabberd_config.hrl").
-include("eldap/eldap.hrl").

-export([init/0]).

-record(ldap, {host,
                eldap_id,
                servers,
                backups,
                port,
                dn,
                password,
                base,
                domains,
                dfilter,
                dsfilter,
                dn_filter,
                dn_filter_attrs
               }).

-define(LDAP_SEARCH_TIMEOUT, 5). % Timeout for LDAP search queries in seconds
-define(RETRY_TIMEOUT, 5000).
-define(BIND_TIMEOUT, 10000).
-define(CMD_TIMEOUT, 5000).

init() ->
    Ldap = parse_options(),
    eldap:start_link(Ldap#ldap.eldap_id,
                     Ldap#ldap.servers,
                     Ldap#ldap.port,
                     Ldap#ldap.dn,
                     Ldap#ldap.password,
                     ""
		),
    DOMAINs = Ldap#ldap.domains,
    SortedDNAttrs = eldap_utils:usort_attrs(["dc"]),
    case eldap_filter:parse(Ldap#ldap.dsfilter) of
        {ok, Filter} ->
            case eldap:search(Ldap#ldap.eldap_id, [{base, Ldap#ldap.base}, {filter, Filter}, {attributes, SortedDNAttrs}]) of
                DOM = #eldap_search_result{entries = Entries} ->
                    lists:flatmap(
                         fun(#eldap_entry{attributes = Attrs, object_name = DN}) ->
                                              case eldap_utils:find_ldap_attrs(DOMAINs, Attrs) of
                                                  "" -> [];
                                                  {Domain, DOMAINFormat} ->
                                                              case jlib:nodeprep(Domain) of
                                                                  error -> [];
                                                                  LD -> [ LD ]
                                                              end
                                              end
                             
                          end, Entries);
                  _ ->
                        []
		end;
             _ ->
                 [];
            
        _ ->
            false
    end.

%stop_odbc(Host) ->
%    Proc = gen_mod:get_module_proc(Host, ejabberd_odbc_sup),
%    supervisor:terminate_child(ejabberd_sup, Proc),
%    supervisor:delete_child(ejabberd_sup, Proc).


normalize(A) ->
    jlib:nodeprep(A).

parse_options() ->
    Host = "default",
    Eldap_ID = Host,
    LDAPServers = ejabberd_config:get_local_option(domain_ldap_host),
    LDAPBackups = case ejabberd_config:get_local_option(domain_ldap_backups) of
                   undefined -> [];
                   Backups -> Backups
                   end,
    LDAPPort = case ejabberd_config:get_local_option(domain_ldap_port) of
                   undefined -> 389;
                   P -> P
               end,
    RootDN = case ejabberd_config:get_local_option(domain_ldap_rootdn) of
                 undefined -> "";
                 RDN -> RDN
             end,
    Password = case ejabberd_config:get_local_option(domain_ldap_password) of
                   undefined -> "";
                   Pass -> Pass
               end,
    DOMAINs = case ejabberd_config:get_local_option(domain_ldap_domains) of
               undefined -> [{"dc", "%d"}];
               DUI -> eldap_utils:uids_domain_subst(Host, DUI)
           end,
    DomainSubFilter = lists:flatten(eldap_utils:generate_subfilter(DOMAINs)),
    DomainFilter = case ejabberd_config:get_local_option(domain_ldap_filter) of
                     undefined -> DomainSubFilter;
                     "" -> DomainSubFilter;
                     DF -> "(&" ++ DomainSubFilter ++ DF ++ ")"
                 end,
    DomainSearchFilter = eldap_filter:do_sub(DomainFilter, [{"%d", "*"}]),

    LDAPBase = ejabberd_config:get_local_option(domain_ldap_base),

    {DNFilter, DNFilterAttrs} =
        case ejabberd_config:get_local_option({ldap_dn_filter, Host}) of
            undefined -> {undefined, undefined};
            {DNF, DNFA} -> {DNF, DNFA}
        end,

    #ldap{host = Host,
           eldap_id = Eldap_ID,
           servers = LDAPServers,
           backups = LDAPBackups,
           port = LDAPPort,
           dn = RootDN,
           password = Password,
           base = LDAPBase,
           domains = DOMAINs,
           dfilter = DomainFilter,
           dsfilter = DomainSearchFilter,
           dn_filter = DNFilter,
           dn_filter_attrs = DNFilterAttrs

          }.

